//*************************************************************************************************
//
//	Description:
//		VisibilityProbe.fx - Shader for doing depth buffer sampling to evaluate visibility.
//
//	<P> Copyright (c) 2008 Blimey! Games Ltd. All rights reserved.
//
//*************************************************************************************************

#include "stddefs.fxh"

//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Textures
//

// Depth buffer in texture form.

texture depthTexture : TEXTURE
<
	string UIName = "Depth Buffer Texture";
	bool appEdit = true;
>;


//
// Samplers
//

// Depth buffer in texture form.

sampler2D depthMap : SAMPLER
<
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture = "depthTexture"; 
	string MinFilter = "Point";
	string MagFilter = "Point";
	string MipFilter = "None";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
>
= sampler_state
{
	Texture = < depthTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	AddressU  = Clamp;
	AddressV  = Clamp;
#if defined(_PS3_)
	LODBias = 0;
#else		// ! _PS3_
	MipMapLODBias = 0;
#endif	// ! _PS3_
	SET_NO_ANISOTROPY
#endif	// SET_FX_SAMPLER_STATES
};


//-----------------------------------------------------------------------
//
// Vertex Shader
//

// Input structure.

struct VSINPUT
{
	float2 position : POSITION;											// Render target space position.
	float3 texCoord : TEXCOORD0;										// Screen space position (UV) and depth (W).
};


// Output structure.

struct VSOUTPUT
{
	float4 position	: POSITION;											// Render target space position.
	float3 texCoord : TEXCOORD0;										// Screen space position (UV) and depth (W).
};


// Main vertex shader.

VSOUTPUT SampleVertexShader( VSINPUT input )
{
	VSOUTPUT output;

	// Expand the input position.

	output.position = float4( input.position.xy, 0.0f, 1.0f );

	// Copy the supplied UVW.

	output.texCoord = input.texCoord;

	return output;
}


//-----------------------------------------------------------------------
//
// Fragment Shader
//

// Platform specific conversion of depth buffer into depth value.
// Matches DepthOfField.fx.

float	ComputeDepth( float4 depthTex )
{
	const float TWO_0 = 1.0f;
	const float TWO_8 = 256.0f;
	const float TWO_16 = 65536.0f;
	const float TWO_24 = 16777216.0f;

#if defined(_PS3_)

	const float3 FACTORS = float3( ( TWO_16 / ( TWO_24 - 1.0f ) ),
																 ( TWO_8  / ( TWO_24 - 1.0f ) ),
																 ( TWO_0  / ( TWO_24 - 1.0f ) ) );
	float3 rescaled = round( float3( depthTex.a, depthTex.r, depthTex.g ) * 255.0f );
	float depth = dot( rescaled, FACTORS );	
	return depth;

#elif defined(_XBOX)

	float	depth = depthTex.x;
	return depth;

#elif defined(_DEPTH_FROM_ZBUFFER_)

	const float3 FACTORS = float3( ( ( TWO_8 - 1.0f ) / TWO_8 ),
																 ( ( TWO_8 - 1.0f ) / TWO_16 ),
																 ( ( TWO_8 - 1.0f ) / TWO_24 ) );
	float depth = dot( depthTex.xyz, FACTORS );
	return depth;

#else

	float	depth = depthTex.x;
	return depth;

#endif
}


// Main fragment shader.

float4 SampleFragmentShader( VSOUTPUT input ) : COLOR0
{
	// Sample and reconstruct depth.

	float4 depthValues = tex2D( depthMap, input.texCoord.xy );
	float depth = ComputeDepth( depthValues );

	// Special case handling - depth of zero is for sky.
	// Sky is guaranteed to be behind everything,
	// apart from stuff artificially put behind everything, including sky.

	float realDepth = ( ( depth == 0.0f ) ? 1.0f : depth );

	// Normal stuff must be nearer than the depth buffer.
	// Normal stuff is nearer than sky.
	// Special stuff is behind anything from depth buffer.

	float isVisible = ( ( input.texCoord.z < realDepth ) ? 1.0f : 0.0f );

	// You can tune the "becoming visible" curve separately from "becoming invisible".
	//
	// Because the factors are logarithmic, which may be unintuitive, consult the cheat sheet:
	// 0.1f - 51 frames to fully apply the change (from 0 to 1 or from 1 to 0).
	// 0.2f - 24 frames
	// 0.3f - 15 frames
	// 0.4f - 11 frames
	// 0.5f -  8 frames
	// 0.6f -  6 frames
	// 0.7f -  5 frames
	// 0.8f -  4 frames
	// 0.9f -  3 frames
	// 0.95f - 2 frames
	// 1.0f -  1 frame
	//
	const float HYSTERESIS_VISIBLE = 0.95f;
	const float HYSTERESIS_INVISIBLE = 0.5f;

	return	( isVisible > 0.0f )
					? float4( 1.0f, 1.0f, 1.0f, HYSTERESIS_VISIBLE )
				 	: float4( 0.0f, 0.0f, 0.0f, HYSTERESIS_INVISIBLE );
}


//-----------------------------------------------------------------------
//
// Technique
//

technique Sample
<
	string normalBehaviour	= "ERMB_RENDER";
	string normalTechnique	= "Sample";
	int    normalDeferredID	= 0;
	bool   usefullprecision = true;
>
{
	pass Pass0
	{
		ZEnable = false;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
#if defined(_PS3_)
		CullFaceEnable = false;
		BlendFunc = int2( SrcAlpha, OneMinusSrcAlpha );
		BlendEquation = int( FuncAdd );
#else		// ! _PS3_
		CullMode = None;
		SrcBlend = SRCALPHA;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif	// ! _PS3_

#if defined(_PS3_)
		VertexShader = compile sce_vp_rsx SampleVertexShader();
		PixelShader = compile sce_fp_rsx SampleFragmentShader();
#else
		VertexShader = compile vs_3_0 SampleVertexShader();
		PixelShader = compile ps_3_0 SampleFragmentShader();
#endif
	}
}
